"use strict";

Polymer({
  is: "gobstones-editor",
  behaviors: [Polymer.BusListenerBehavior, Polymer.LocalizationBehavior],
  properties: {
    mode: { type: String, value: "main" },
    code: {
      type: Object,
      value: { main: "", library: "", teacher: "" }
    },
    fontSize: {
      type: Number,
      observer: "_fontSizeChanged"
    },
    readonly: {
      type: Boolean,
      value: false
    },
    primitiveProcedures: {
      type: Array,
      value: []
    },
    primitiveFunctions: {
      type: Array,
      value: []
    },
    isShowingProceduresHint: {
      type: Boolean,
      value: false
    },
    withRunner: {
      type: Boolean,
      value: false
    }
  },
  listeners: {
    "ace.editor-ready": "onAceReady",
    "ace.editor-content": "onContentChange"
  },

  ready: function ready() {
    var _this = this;

    this.MIN_FONT_SIZE = 8;
    this.DEFAULT_FONT_SIZE = 17;
    this.HOP_SIZE = 3;

    if (this.withRunner) {
      var boardsPanel = document.getElementById("boards");
      if (boardsPanel) {
        this.runner = boardsPanel.$.runner;
        this.runner.addEventListener("run", function (_ref) {
          var detail = _ref.detail;

          _this._onRunRequest(detail);
          _this.readonly = true;
        });
        this.runner.addEventListener("cancel", function () {
          _this.readonly = false;

          if (_this.isInteractiveOn) {
            _this.isInteractiveOn = false;
            return;
          }
          window.BUS.fire("cancel-request");
        });
        this.runner.addEventListener("end", function () {
          _this.readonly = false;
        });
      }

      this.subscribeTo("initial-state", function (event) {
        _this._runCode(event);
      }).subscribeTo("interactive-run", function () {
        _this.isInteractiveOn = true;
      });
    }

    this.editor = this.$.ace.editor;
    this._setFatalities();

    this.stylist = new Stylist();
    this._removePrintMarginColumn();
    $(window).resize(function () {
      _this._fixEditorHeight();
    });

    setTimeout(function () {
      if (window.LOAD_PENDING_PROJECT) {
        window.LOAD_PENDING_PROJECT();
        window.LOAD_PENDING_PROJECT = undefined;
      }
    }, 0);
  },

  setCode: function setCode(code) {
    var mode = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : "main";
    var withTeacherErrorsReport = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;

    this.code[mode] = code;
    if (this.mode === mode) this.editor.setValue(code);

    if (mode === "teacher") {
      this._onTeacherLibraryChange(code, withTeacherErrorsReport);
    }
  },

  onAceReady: function onAceReady() {
    this.$.ace.editor.$blockScrolling = Infinity;
  },

  onContentChange: function onContentChange(content) {
    this.set("code." + this.mode, content.detail.value);
    this.setAsDirty();
    this.fire("content-change");
  },

  increaseFontSize: function increaseFontSize() {
    this.fontSize += this.HOP_SIZE;
  },

  decreaseFontSize: function decreaseFontSize() {
    this.fontSize -= this.HOP_SIZE;
  },

  toggleMode: function toggleMode() {
    this._setMode(this.mode === "main" ? "library" : "main");
  },

  setAsDirty: function setAsDirty() {
    this.editor.getSession().setAnnotations([]);
    window.BUS.fire("editor-dirty");
  },

  reset: function reset() {
    this._setMode("main");
    this.setCode("", "main");
    this.setCode("", "library");
    this.setCode("", "teacher");
  },

  _onRunRequest: function _onRunRequest(options) {
    this._setMode("main");
    window.BUS.fire("run-request", options);
  },

  _initializeDefaultFontSize: function _initializeDefaultFontSize() {
    this.fontSize = this.DEFAULT_FONT_SIZE;
  },

  _fontSizeChanged: function _fontSizeChanged(newValue, oldValue) {
    var _this2 = this;

    if (newValue <= this.MIN_FONT_SIZE) {
      this.fontSize = oldValue;
      return;
    }

    this.editor.setFontSize(newValue);
    setTimeout(function () {
      _this2._fixEditorHeight();
    }, 500);
  },

  _runCode: function _runCode(_ref2) {
    var _this3 = this;

    var initialState = _ref2.initialState,
        controller = _ref2.controller;
    var code = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.code;

    this.editor.getSession().clearAnnotations();

    try {
      controller.start({ initialState: initialState, code: code }, {
        onCompilationError: function onCompilationError(e) {
          if (e.message === "missing-program") e.message = _this3.localize("missing-program");

          _this3._reportError(e, "error");
          window.BUS.fire("compilation-error");
        },
        onTeacherCompilationError: function onTeacherCompilationError(e) {
          _this3.runner.reportTeacherLibraryErrors(e);
        },
        onInteractiveRun: function onInteractiveRun() {
          window.BUS.fire("interactive-run");
        },
        onResult: function onResult(state) {
          return _this3._notify(state);
        },
        onReturnValue: function onReturnValue(value, actualReturnValue) {
          window.BUS.fire("return-value", { value: { value: value, actualReturnValue: actualReturnValue } });
        }
      }, this.runner.speed);
    } catch (e) {
      window.BUS.fire("unknown-error", e);
      console.error("---UNKNOWN ERROR---");
      throw e;
    }
  },

  checkCompilationErrors: function checkCompilationErrors() {
    var _this4 = this;

    this.editor.getSession().clearAnnotations();

    var parser = this.runner.gbsCodeRunner.parser;
    var code = { main: "program {}\n" + this.code.main, library: "program {}" };
    parser.parse(code.main, function (e) {
      e.location = parser.getErrorLineAndMode(e, code, true);
      _this4._reportError(e, "error", true);
    }, function () {});
  },

  _notify: function _notify(state) {
    if (state.error) {
      this._reportError(state.error, "info");
      window.BUS.fire("execution-error", state.error.message);
    } else window.BUS.fire("execution-result", { board: state });
  },

  _reportError: function _reportError(error, type, onTheFly) {
    if (error.location.mode === "library") this._setMode("library");

    this._setAnnotation(error, type, onTheFly);
  },

  _setAnnotation: function _setAnnotation(error, type) {
    var onTheFly = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;

    var row = error.location.line - 1;

    if (!onTheFly) {
      this.editor.selection.setRange({ start: { row: row, column: 0 }, end: { row: row, column: Infinity } });
      this.editor.scrollToLine(row, true, true, function () {});
    }

    this.editor.getSession().setAnnotations([{
      row: row,
      column: 0,
      text: error.message,
      type: type
    }]);
  },


  _setMode: function _setMode(mode) {
    this.mode = mode;
    window.BUS.fire("mode-change", this.mode);
    this.editor.setValue(this.code[this.mode]);
    this.editor.focus();
  },

  _onTeacherLibraryChange: function _onTeacherLibraryChange(teacherCode) {
    var withErrorReport = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    this.primitiveProcedures = [];
    this.primitiveFunctions = [];
    window.BUS.fire("primitive-actions-off");

    try {
      var actions = new Parser().getActionNamesFromSource(teacherCode);

      if (!_.isEmpty(actions.procedureNames) || !_.isEmpty(actions.functionNames)) {
        this.primitiveProcedures = actions.procedureNames;
        this.primitiveFunctions = actions.functionNames;

        window.BUS.fire("primitive-actions-on");
      }
    } catch (e) {
      if (withErrorReport) this.runner.reportTeacherLibraryErrors(e);
    }
  },

  _removePrintMarginColumn: function _removePrintMarginColumn() {
    this.editor.setShowPrintMargin(false);
  },

  _setFatalities: function _setFatalities() {
    var _this5 = this;

    var ace = this.$.ace;

    ace.editor.commands.addCommand({
      name: "run-code",
      bindKey: { win: "ctrl+enter", mac: "command+enter" },
      exec: function exec() {
        _this5.runner.requestRun();
      }
    });

    var key = "Control";
    $(window).keydown(function (event) {
      if (event.key === key) _this5.isShowingProceduresHint = true;
    });

    $(window).keyup(function (event) {
      if (event.key === key) _this5.isShowingProceduresHint = false;
    });
  },

  _fixEditorHeight: function _fixEditorHeight() {
    this.stylist.correctEditorHeight(this.editor);
  },

  _shouldShowProceduresHint: function _shouldShowProceduresHint(primitiveProcedures, primitiveFunctions) {
    return !_.isEmpty(primitiveProcedures) || !_.isEmpty(primitiveFunctions);
  }
});